<?php
/**
 * Plugin Name: Massblogger PartnerAds
 * Plugin URI: https://massblogger.com
 * Description: Import PartnerAds product feeds and display them with shortcodes. Tag products, configure design, embed anywhere.
 * Version: 1.2.0
 * Author: Massblogger
 * Author URI: https://massblogger.com
 * License: GPL v2 or later
 * Text Domain: massblogger-partnerads
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) exit;
define( 'MBPA_VERSION', '1.2.0' );

// =====================================================
// POST TYPE + TAXONOMY (tags)
// =====================================================

add_action( 'init', 'mbpa_register' );
function mbpa_register() {
    register_post_type( 'mbpa_product', array(
        'labels' => array(
            'name' => 'PartnerAds', 'singular_name' => 'Product', 'menu_name' => 'PartnerAds',
            'all_items' => 'Products', 'search_items' => 'Search Products',
        ),
        'public'             => false,
        'show_ui'            => true,
        'show_in_menu'       => true,
        'menu_icon'          => 'dashicons-cart',
        'supports'           => array( 'title', 'thumbnail' ),
        'capabilities'       => array(
            'create_posts' => 'do_not_allow',
        ),
        'map_meta_cap'       => true,
        'has_archive'        => false,
        'rewrite'            => false,
    ) );

    register_taxonomy( 'mbpa_tag', 'mbpa_product', array(
        'labels' => array(
            'name' => 'Product Tags', 'singular_name' => 'Tag',
            'search_items' => 'Search Tags', 'all_items' => 'All Tags',
            'edit_item' => 'Edit Tag', 'add_new_item' => 'Add Tag',
        ),
        'hierarchical' => false,
        'show_ui'      => true,
        'show_in_menu' => true,
        'rewrite'      => false,
        'public'       => false,
        'show_admin_column' => true,
    ) );
}

// =====================================================
// LIFECYCLE
// =====================================================

register_activation_hook( __FILE__, function() {
    mbpa_register();
    flush_rewrite_rules();
    if ( ! wp_next_scheduled( 'mbpa_cron_sync' ) )
        wp_schedule_event( time(), 'twicedaily', 'mbpa_cron_sync' );
});
register_deactivation_hook( __FILE__, function() {
    wp_clear_scheduled_hook( 'mbpa_cron_sync' );
});

// =====================================================
// ADMIN COLUMNS
// =====================================================

add_filter( 'manage_mbpa_product_posts_columns', function( $c ) {
    $n = array();
    foreach ( $c as $k => $v ) {
        if ( $k === 'title' ) {
            $n['mbpa_image'] = '';
            $n[ $k ] = $v;
            $n['mbpa_sku']   = 'SKU';
            $n['mbpa_price'] = 'Price';
            $n['mbpa_brand'] = 'Brand';
        } else {
            $n[ $k ] = $v;
        }
    }
    return $n;
});

add_action( 'manage_mbpa_product_posts_custom_column', function( $col, $id ) {
    if ( $col === 'mbpa_image' ) {
        $img = get_post_meta( $id, '_mbpa_image', true );
        if ( $img ) echo '<img src="' . esc_url( $img ) . '" style="width:44px;height:44px;object-fit:contain;border-radius:4px;background:#f5f5f5;padding:2px">';
    } elseif ( $col === 'mbpa_sku' ) {
        $sku = get_post_meta( $id, '_mbpa_sku', true );
        echo $sku ? '<code style="font-size:11px;background:#f0f0f1;padding:2px 6px;border-radius:3px">' . esc_html( $sku ) . '</code>' : '–';
    } elseif ( $col === 'mbpa_price' ) {
        $p = get_post_meta( $id, '_mbpa_price', true );
        $old = get_post_meta( $id, '_mbpa_old_price', true );
        $c = get_post_meta( $id, '_mbpa_currency', true ) ?: 'DKK';
        if ( $p ) {
            echo '<strong style="color:#d63031">' . esc_html( $p . ' ' . $c ) . '</strong>';
            if ( $old ) echo '<br><span style="color:#999;text-decoration:line-through;font-size:12px">' . esc_html( $old . ' ' . $c ) . '</span>';
        } else { echo '–'; }
    } elseif ( $col === 'mbpa_brand' ) {
        echo esc_html( get_post_meta( $id, '_mbpa_brand', true ) ?: '–' );
    }
}, 10, 2 );

add_filter( 'manage_edit-mbpa_product_sortable_columns', function( $c ) {
    $c['mbpa_sku'] = 'mbpa_sku';
    $c['mbpa_price'] = 'mbpa_price';
    $c['mbpa_brand'] = 'mbpa_brand';
    return $c;
});

add_action( 'pre_get_posts', function( $q ) {
    if ( ! is_admin() || ! $q->is_main_query() || $q->get('post_type') !== 'mbpa_product' ) return;
    $ob = $q->get('orderby');
    if ( $ob === 'mbpa_sku' ) { $q->set('meta_key','_mbpa_sku'); $q->set('orderby','meta_value'); }
    elseif ( $ob === 'mbpa_price' ) { $q->set('meta_key','_mbpa_price'); $q->set('orderby','meta_value_num'); }
    elseif ( $ob === 'mbpa_brand' ) { $q->set('meta_key','_mbpa_brand'); $q->set('orderby','meta_value'); }
});

add_action( 'admin_head', function() {
    $s = get_current_screen();
    if ( $s && $s->post_type === 'mbpa_product' ) {
        echo '<style>.column-mbpa_image{width:54px} .column-mbpa_sku{width:100px} .page-title-action{display:none!important}</style>';
    }
});

// =====================================================
// ADMIN MENU
// =====================================================

add_action( 'admin_menu', function() {
    add_submenu_page( 'edit.php?post_type=mbpa_product', 'Settings', 'Settings', 'manage_options', 'mbpa-settings', 'mbpa_settings_page' );
});

add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function( $l ) {
    array_unshift( $l, '<a href="' . admin_url( 'edit.php?post_type=mbpa_product&page=mbpa-settings' ) . '">Settings</a>' );
    return $l;
});

// =====================================================
// FEED PARSING (compact)
// =====================================================

function mbpa_xf( $xml, $f ) {
    $q = preg_quote( $f, '/' );
    if ( preg_match( "/<{$q}[^>]*><!\[CDATA\[([\\s\\S]*?)\]\]><\/{$q}>/i", $xml, $m ) ) return trim( $m[1] );
    if ( preg_match( "/<{$q}[^>]*>([^<]*)<\/{$q}>/i", $xml, $m ) ) return trim( $m[1] );
    return null;
}

function mbpa_fmap() {
    return array(
        'id'       => array('id','sku','SKU','varenummer','produktid','product_id'),
        'name'     => array('produktnavn','name','title','product_name','navn'),
        'desc'     => array('beskrivelse','description','short_description','produktbeskrivelse'),
        'price'    => array('nypris','price','sale_price','pris'),
        'oldprice' => array('forpris','regular_price','original_price','old_price','vejlpris'),
        'currency' => array('currency','priceCurrency','valuta'),
        'image'    => array('billedurl','billede_url','image','imageUrl','image_url','image_link','billede','produktbillede'),
        'url'      => array('vareurl','url','link','product_url','productUrl','tracking_url','deeplink','affiliate_url','produkturl'),
        'category' => array('kategori','category','product_category','produktkategori'),
        'brand'    => array('brand','maerke','manufacturer','vendor','producent'),
        'stock'    => array('lager','lagerstatus','availability','in_stock','stock'),
        'ean'      => array('ean','EAN','gtin','GTIN','barcode','stregkode'),
    );
}

function mbpa_parse_xml( $t ) {
    $out = array(); $map = mbpa_fmap();
    if ( ! preg_match_all( '/<(?:product|item|entry|row|produkt)[^>]*>([\s\S]*?)<\/(?:product|item|entry|row|produkt)>/i', $t, $ms ) ) return $out;
    foreach ( $ms[1] as $px ) {
        $p = array();
        foreach ( $map as $k => $fs ) { foreach ( $fs as $f ) { $v = mbpa_xf( $px, $f ); if ( $v !== null && ! isset( $p[$k] ) ) { $p[$k] = $v; break; } } }
        foreach ( array('id'=>'g:id','name'=>'g:title','desc'=>'g:description','url'=>'g:link','image'=>'g:image_link','price'=>'g:price','brand'=>'g:brand','ean'=>'g:gtin') as $k => $f ) {
            if ( ! isset( $p[$k] ) ) { $v = mbpa_xf( $px, $f ); if ( $v ) $p[$k] = $v; }
        }
        if ( ! empty( $p['name'] ) || ! empty( $p['id'] ) ) $out[] = $p;
    }
    return $out;
}

function mbpa_parse_csv( $t, $d = ',' ) {
    $ls = explode( "\n", trim( $t ) );
    if ( count( $ls ) < 2 ) return array();
    $hs = array_map( function($h) { return strtolower( trim( $h, " \t\r\n\"'" ) ); }, explode( $d, $ls[0] ) );
    $map = mbpa_fmap(); $hm = array();
    foreach ( $hs as $i => $h ) { foreach ( $map as $k => $fs ) { if ( in_array( $h, array_map('strtolower',$fs), true ) ) { $hm[$i] = $k; break; } } }
    $out = array();
    for ( $i = 1; $i < count($ls); $i++ ) {
        $vs = array(); $c = ''; $q = false;
        for ( $j = 0; $j < strlen($ls[$i]); $j++ ) {
            $ch = $ls[$i][$j];
            if ( $ch === '"' ) $q = !$q;
            elseif ( $ch === $d && !$q ) { $vs[] = trim($c," \t\r\n\"'"); $c = ''; }
            else $c .= $ch;
        }
        $vs[] = trim($c," \t\r\n\"'");
        if ( count($vs) < 3 ) continue;
        $p = array();
        foreach ( $hm as $idx => $k ) { if ( isset($vs[$idx]) && $vs[$idx] !== '' && !isset($p[$k]) ) $p[$k] = $vs[$idx]; }
        if ( !empty($p['name']) || !empty($p['id']) ) $out[] = $p;
    }
    return $out;
}

function mbpa_fetch( $url ) {
    $r = wp_remote_get( $url, array( 'timeout' => 120, 'user-agent' => 'Mozilla/5.0 (compatible; MassbloggerPA/1.0)' ) );
    if ( is_wp_error($r) ) return $r;
    if ( wp_remote_retrieve_response_code($r) !== 200 ) return new WP_Error('http', 'HTTP ' . wp_remote_retrieve_response_code($r));
    $b = wp_remote_retrieve_body($r);
    if ( empty($b) ) return new WP_Error('empty','Empty response');
    $t = trim($b); $raw = array();
    if ( strpos($t,'<?xml') === 0 || strpos($t,'<') === 0 ) $raw = mbpa_parse_xml($b);
    elseif ( strpos($t,'{') === 0 || strpos($t,'[') === 0 ) { $j = json_decode($b,true); if(is_array($j)) $raw = isset($j[0]) ? $j : ($j['products'] ?? $j['items'] ?? $j['data'] ?? array()); }
    else { $fl = strtok($b,"\n"); $d = strpos($fl,"\t")!==false ? "\t" : (strpos($fl,';')!==false ? ';' : ','); $raw = mbpa_parse_csv($b,$d); }
    if ( empty($raw) ) return new WP_Error('parse','No products found');
    return array_map( function($p) {
        $pr = isset($p['price']) ? str_replace(',','.',preg_replace('/[^0-9.,]/','',$p['price'])) : '';
        $op = isset($p['oldprice']) ? str_replace(',','.',preg_replace('/[^0-9.,]/','',$p['oldprice'])) : '';
        return array( 'sku'=>$p['id']??'', 'name'=>$p['name']??'', 'desc'=>$p['desc']??'', 'price'=>$pr, 'old_price'=>($op&&$op!==$pr)?$op:'', 'currency'=>$p['currency']??'DKK', 'image'=>$p['image']??'', 'url'=>$p['url']??'', 'category'=>$p['category']??'', 'brand'=>$p['brand']??'', 'ean'=>$p['ean']??'' );
    }, $raw );
}

// =====================================================
// IMPORT
// =====================================================

function mbpa_import( $feed_url ) {
    $products = mbpa_fetch( $feed_url );
    if ( is_wp_error($products) ) return $products;
    $st = array( 'created'=>0, 'updated'=>0, 'errors'=>0, 'total'=>count($products) );
    foreach ( $products as $d ) {
        try {
            $sku = sanitize_text_field( $d['sku'] );
            $eid = 0;
            if ( $sku ) { $q = get_posts(array('post_type'=>'mbpa_product','meta_key'=>'_mbpa_sku','meta_value'=>$sku,'numberposts'=>1,'fields'=>'ids')); if(!empty($q)) $eid=$q[0]; }
            if ( !$eid && $d['url'] ) { $q = get_posts(array('post_type'=>'mbpa_product','meta_key'=>'_mbpa_url','meta_value'=>$d['url'],'numberposts'=>1,'fields'=>'ids')); if(!empty($q)) $eid=$q[0]; }
            $new = !$eid;
            $pa = array( 'post_type'=>'mbpa_product', 'post_title'=>sanitize_text_field($d['name']), 'post_content'=>wp_kses_post($d['desc']), 'post_status'=>'publish' );
            if ( $eid ) { $pa['ID'] = $eid; wp_update_post($pa); $pid = $eid; }
            else { $pid = wp_insert_post($pa); if(is_wp_error($pid)){$st['errors']++;continue;} }
            update_post_meta($pid,'_mbpa_sku',$sku);
            update_post_meta($pid,'_mbpa_url',esc_url_raw($d['url']));
            update_post_meta($pid,'_mbpa_price',$d['price']);
            update_post_meta($pid,'_mbpa_old_price',$d['old_price']);
            update_post_meta($pid,'_mbpa_currency',$d['currency']);
            update_post_meta($pid,'_mbpa_brand',sanitize_text_field($d['brand']));
            update_post_meta($pid,'_mbpa_image',esc_url_raw($d['image']));
            update_post_meta($pid,'_mbpa_ean',sanitize_text_field($d['ean']));
            update_post_meta($pid,'_mbpa_feed',esc_url_raw($feed_url));
            // Auto-tag by category from feed
            if ( !empty($d['category']) ) {
                $parts = array_filter(array_map('trim', preg_split('/[>\/|]/', $d['category'])));
                if ( !empty($parts) ) wp_set_object_terms( $pid, $parts, 'mbpa_tag', true );
            }
            $st[$new?'created':'updated']++;
        } catch(\Exception $e) { $st['errors']++; }
    }
    $log = get_option('mbpa_log',array());
    $log[] = array('feed'=>$feed_url,'time'=>current_time('mysql'),'stats'=>$st);
    if(count($log)>20) $log = array_slice($log,-20);
    update_option('mbpa_log',$log);
    return $st;
}

// =====================================================
// CRON
// =====================================================

add_action( 'mbpa_cron_sync', function() {
    $s = get_option('mbpa_settings',array());
    if ( empty($s['auto_sync']) ) return;
    $urls = array_filter(array_map('trim',explode("\n",get_option('mbpa_feeds',''))));
    foreach($urls as $u) { if(filter_var($u,FILTER_VALIDATE_URL)) mbpa_import($u); }
});

// =====================================================
// AJAX
// =====================================================

add_action('wp_ajax_mbpa_save', function() {
    check_ajax_referer('mbpa_nonce','nonce');
    if(!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
    if(isset($_POST['feeds'])) update_option('mbpa_feeds', sanitize_textarea_field($_POST['feeds']));
    // Merge with existing settings so tab-based saves don't wipe other tabs
    $existing = get_option('mbpa_settings', array());
    $new = array();
    $bools = array('auto_sync','show_brand','show_image','show_arrow','table_show_header','table_striped','table_hover','list_show_btn','grid_shadow');
    $colors = array('btn_color','btn_text_color','price_color','card_border','table_hover_color');
    $ints = array('btn_radius','grid_name_size','grid_price_size','grid_btn_size','grid_img_ratio','table_name_size','table_price_size','table_btn_size','table_row_padding','box_name_size','box_price_size','box_desc_lines','box_img_size','list_name_size','list_price_size');
    $texts = array('button_text','table_btn_style','table_border');
    $fields = array_merge($bools,$colors,$ints,$texts);
    foreach($fields as $f) {
        if(isset($_POST[$f])) {
            if(in_array($f,$bools)) $new[$f] = !empty($_POST[$f]);
            elseif(in_array($f,$colors)) $new[$f] = sanitize_hex_color($_POST[$f]);
            elseif(in_array($f,$ints)) $new[$f] = intval($_POST[$f]);
            else $new[$f] = sanitize_text_field($_POST[$f]);
        }
    }
    update_option('mbpa_settings', array_merge($existing, $new));
    wp_send_json_success('Saved');
});

add_action('wp_ajax_mbpa_preview', function() {
    check_ajax_referer('mbpa_nonce','nonce');
    if(!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
    $u = esc_url_raw($_POST['url']??'');
    if(empty($u)) wp_send_json_error('URL required');
    $p = mbpa_fetch($u);
    if(is_wp_error($p)) wp_send_json_error($p->get_error_message());
    wp_send_json_success(array('total'=>count($p),'sample'=>array_slice($p,0,5)));
});

add_action('wp_ajax_mbpa_import', function() {
    check_ajax_referer('mbpa_nonce','nonce');
    if(!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
    set_time_limit(600);
    $urls = array_filter(array_map('trim',explode("\n",get_option('mbpa_feeds',''))));
    if(empty($urls)) wp_send_json_error('No feed URLs');
    $t = array('created'=>0,'updated'=>0,'errors'=>0,'total'=>0,'feeds'=>0);
    foreach($urls as $u) {
        if(!filter_var($u,FILTER_VALIDATE_URL)) continue;
        $r = mbpa_import($u); if(is_wp_error($r)){$t['errors']++;continue;}
        $t['feeds']++;
        foreach(array('created','updated','errors','total') as $k) $t[$k]+=$r[$k];
    }
    wp_send_json_success($t);
});

add_action('wp_ajax_mbpa_delete_all', function() {
    check_ajax_referer('mbpa_nonce','nonce');
    if(!current_user_can('manage_options')) wp_send_json_error('Unauthorized');
    $ps = get_posts(array('post_type'=>'mbpa_product','numberposts'=>-1,'fields'=>'ids'));
    $n=0; foreach($ps as $id){wp_delete_post($id,true);$n++;}
    wp_send_json_success(array('deleted'=>$n));
});

// =====================================================
// DESIGN DEFAULTS
// =====================================================

function mbpa_design() {
    $s = get_option('mbpa_settings', array());
    return array(
        'btn_color'      => $s['btn_color'] ?? '#2d2d2d',
        'btn_text_color' => $s['btn_text_color'] ?? '#ffffff',
        'btn_radius'     => $s['btn_radius'] ?? 6,
        'price_color'    => $s['price_color'] ?? '#d63031',
        'show_brand'     => isset($s['show_brand']) ? $s['show_brand'] : true,
        'show_image'     => isset($s['show_image']) ? $s['show_image'] : true,
        'show_arrow'     => isset($s['show_arrow']) ? $s['show_arrow'] : true,
        'card_border'    => $s['card_border'] ?? '#e5e5e5',
        'button_text'    => $s['button_text'] ?? 'Se produkt',
        // Grid
        'grid_name_size'  => $s['grid_name_size'] ?? 14,
        'grid_price_size' => $s['grid_price_size'] ?? 16,
        'grid_btn_size'   => $s['grid_btn_size'] ?? 13,
        'grid_shadow'     => isset($s['grid_shadow']) ? $s['grid_shadow'] : true,
        // Table
        'table_name_size'    => $s['table_name_size'] ?? 14,
        'table_price_size'   => $s['table_price_size'] ?? 16,
        'table_btn_size'     => $s['table_btn_size'] ?? 12,
        'table_btn_style'    => $s['table_btn_style'] ?? 'pill',
        'table_show_header'  => isset($s['table_show_header']) ? $s['table_show_header'] : true,
        'table_striped'      => isset($s['table_striped']) ? $s['table_striped'] : false,
        'table_border'       => $s['table_border'] ?? 'horizontal',
        'table_hover'        => isset($s['table_hover']) ? $s['table_hover'] : true,
        'table_hover_color'  => $s['table_hover_color'] ?? '#f8f8f8',
        'table_row_padding'  => $s['table_row_padding'] ?? 12,
        // Box
        'box_name_size'  => $s['box_name_size'] ?? 18,
        'box_price_size' => $s['box_price_size'] ?? 20,
        'box_desc_lines' => $s['box_desc_lines'] ?? 3,
        'box_img_size'   => $s['box_img_size'] ?? 160,
        // List
        'list_name_size'  => $s['list_name_size'] ?? 14,
        'list_price_size' => $s['list_price_size'] ?? 15,
        'list_show_btn'   => isset($s['list_show_btn']) ? $s['list_show_btn'] : true,
    );
}

// =====================================================
// SETTINGS PAGE
// =====================================================

function mbpa_settings_page() {
    $feeds = get_option('mbpa_feeds','');
    $s     = get_option('mbpa_settings',array());
    $log   = get_option('mbpa_log',array());
    $cnt   = wp_count_posts('mbpa_product');
    $total = isset($cnt->publish) ? $cnt->publish : 0;
    $d     = mbpa_design();
    $tags  = get_terms(array('taxonomy'=>'mbpa_tag','hide_empty'=>false));
    $tab   = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'settings';
    $base  = admin_url('edit.php?post_type=mbpa_product&page=mbpa-settings');
    ?>
    <style>
        .mp{max-width:880px}
        .mp-h{background:#fff;border:1px solid #c3c4c7;border-radius:6px;padding:14px 20px;margin:16px 0;display:flex;align-items:center;gap:14px}
        .mp-logo{font-size:17px;font-weight:700;color:#1d2327;white-space:nowrap}
        .mp-desc{color:#50575e;font-size:13px;line-height:1.5;border-left:1px solid #dcdcde;padding-left:14px}
        .mp-desc a{color:#2271b1;text-decoration:none}
        .mp-tabs{display:flex;gap:0;border-bottom:1px solid #c3c4c7;margin-bottom:0}
        .mp-tab{padding:10px 20px;font-size:13px;font-weight:500;color:#50575e;text-decoration:none;border:1px solid transparent;border-bottom:none;margin-bottom:-1px;border-radius:4px 4px 0 0;transition:all .15s}
        .mp-tab:hover{color:#1d2327;background:#f6f7f7}
        .mp-tab.active{background:#fff;color:#1d2327;border-color:#c3c4c7;font-weight:600}
        .mp-panel{background:#fff;border:1px solid #c3c4c7;border-top:none;border-radius:0 0 6px 6px;padding:24px;margin-bottom:16px}
        .mp-sec{margin-bottom:32px;padding-bottom:24px;border-bottom:1px solid #eee}
        .mp-sec:last-child{border-bottom:none;margin-bottom:0;padding-bottom:0}
        .mp-sec h2{margin:0 0 4px;font-size:15px;font-weight:600;color:#1d2327}
        .mp-sec .sub{color:#50575e;font-size:13px;margin:0 0 16px}
        .mp-r{margin-bottom:14px}
        .mp-r label{display:block;font-size:13px;font-weight:500;margin-bottom:4px;color:#1d2327}
        .mp-r input[type=text],.mp-r input[type=number],.mp-r textarea,.mp-r select{width:100%;padding:7px 10px;border:1px solid #c3c4c7;border-radius:4px;font-size:13px;background:#fff}
        .mp-r textarea{font-family:monospace}
        .mp-r input:focus,.mp-r textarea:focus,.mp-r select:focus{border-color:#2271b1;box-shadow:0 0 0 1px #2271b1;outline:none}
        .mp-r .h{font-size:12px;color:#787c82;margin-top:3px}
        .mp-ch{display:flex;align-items:center;gap:8px;margin-bottom:10px}
        .mp-ch input[type=checkbox]{width:16px;height:16px}
        .mp-ch label{font-size:13px;margin:0}
        .mp-g2{display:grid;grid-template-columns:1fr 1fr;gap:12px}
        .mp-g3{display:grid;grid-template-columns:1fr 1fr 1fr;gap:12px}
        .mp-g4{display:grid;grid-template-columns:1fr 1fr 1fr 1fr;gap:12px}
        .mp-color{display:flex;align-items:center;gap:8px}
        .mp-color input[type=color]{width:36px;height:30px;border:1px solid #c3c4c7;border-radius:4px;padding:2px;cursor:pointer}
        .mp-color span{font-size:13px;color:#1d2327}
        .mp-btn{display:inline-flex;align-items:center;gap:6px;padding:8px 16px;border-radius:4px;font-size:13px;font-weight:500;cursor:pointer;border:1px solid #c3c4c7;background:#f6f7f7;color:#1d2327;transition:all .15s}
        .mp-btn:hover{background:#f0f0f1;border-color:#8c8f94}
        .mp-bp{background:#2271b1;border-color:#2271b1;color:#fff}
        .mp-bp:hover{background:#135e96;color:#fff}
        .mp-bd{color:#d63638;border-color:#d63638}
        .mp-bd:hover{background:#d63638;color:#fff}
        .mp-btn:disabled{opacity:.5;cursor:not-allowed}
        .mp-acts{display:flex;gap:8px;flex-wrap:wrap;margin-top:16px}
        .mp-stat{display:inline-block;padding:14px 18px;background:#f6f7f7;border:1px solid #dcdcde;border-radius:6px;text-align:center;margin-right:8px;min-width:90px}
        .mp-stat .n{font-size:26px;font-weight:700;color:#1d2327}
        .mp-stat .l{font-size:11px;color:#787c82;margin-top:2px}
        .mp-log{width:100%;border-collapse:collapse;font-size:13px;margin-top:8px}
        .mp-log th,.mp-log td{text-align:left;padding:7px 8px;border-bottom:1px solid #dcdcde}
        .mp-log th{background:#f6f7f7;font-weight:600}
        .mp-badge{display:inline-block;padding:2px 8px;border-radius:10px;font-size:11px;font-weight:600}
        .mp-bg{background:#d4edda;color:#155724}.mp-bb{background:#cce5ff;color:#004085}.mp-br{background:#f8d7da;color:#721c24}
        .mp-tags{display:flex;flex-wrap:wrap;gap:6px;margin-top:8px}
        .mp-tag{background:#f0f0f1;border:1px solid #dcdcde;border-radius:14px;padding:3px 12px;font-size:12px;color:#1d2327}
        .mp-prev{margin-top:16px;padding:20px;background:#fafafa;border:1px solid #dcdcde;border-radius:6px}
        .mp-prev h3{margin:0 0 14px;font-size:12px;font-weight:600;color:#787c82;text-transform:uppercase;letter-spacing:.5px}
        .mp-sep{border:none;border-top:1px solid #dcdcde;margin:18px 0}
        .mp-spin{display:inline-block;width:14px;height:14px;border:2px solid #dcdcde;border-top-color:#2271b1;border-radius:50%;animation:mps .5s linear infinite;vertical-align:middle}
        @keyframes mps{to{transform:rotate(360deg)}}
        #mp-status{margin-top:12px;font-size:13px}
        .mp-ptable{width:100%;border-collapse:collapse;font-size:12px;margin-top:12px}
        .mp-ptable th,.mp-ptable td{padding:6px 8px;border:1px solid #dcdcde;text-align:left}
        .mp-ptable th{background:#f6f7f7}
        .mp-ptable img{max-width:40px;max-height:40px;border-radius:3px}
        code.sc{background:#f0f0f1;padding:3px 8px;border-radius:3px;font-size:12px}
    </style>
    <div class="wrap mp">
        <h1>PartnerAds</h1>
        <div class="mp-h">
            <div class="mp-logo">🛒 PartnerAds</div>
            <div class="mp-desc">
                Import product feeds from <a href="https://www.partnerads.com" target="_blank">PartnerAds</a>, tag them, display anywhere with shortcodes.
                By <a href="https://massblogger.com" target="_blank">Massblogger</a>.
            </div>
        </div>

        <?php if($total > 0): ?>
        <div style="margin-bottom:16px">
            <div class="mp-stat"><div class="n"><?php echo $total; ?></div><div class="l">Products</div></div>
            <div class="mp-stat"><div class="n"><?php echo is_array($tags)?count($tags):0; ?></div><div class="l">Tags</div></div>
        </div>
        <?php endif; ?>

        <div class="mp-tabs">
            <a class="mp-tab <?php echo $tab==='settings'?'active':''; ?>" href="<?php echo esc_url($base.'&tab=settings'); ?>">Settings</a>
            <a class="mp-tab <?php echo $tab==='design'?'active':''; ?>" href="<?php echo esc_url($base.'&tab=design'); ?>">Design</a>
        </div>

        <div class="mp-panel">

        <?php if($tab==='settings'): ?>

            <!-- FEEDS -->
            <div class="mp-sec">
                <h2>Feed URLs</h2>
                <p class="sub">One URL per line. Categories from the feed are auto-applied as tags.</p>
                <div class="mp-r">
                    <textarea id="mp-feeds" rows="4" placeholder="https://www.partnerads.com/da/feed/..."><?php echo esc_textarea($feeds); ?></textarea>
                </div>
                <div class="mp-ch">
                    <input type="checkbox" id="mp-sync" <?php checked(!empty($s['auto_sync'])); ?>>
                    <label for="mp-sync">Auto-sync twice daily</label>
                </div>
                <div class="mp-acts">
                    <button class="mp-btn mp-bp" id="mp-save-feeds">Save</button>
                    <button class="mp-btn mp-bp" id="mp-import">Import All</button>
                    <button class="mp-btn" id="mp-test">Preview Feed</button>
                    <?php if($total>0): ?><button class="mp-btn mp-bd" id="mp-del">Delete All (<?php echo $total; ?>)</button><?php endif; ?>
                </div>
                <div id="mp-status"></div>
                <div id="mp-preview"></div>
            </div>

            <!-- SHORTCODES -->
            <div class="mp-sec">
                <h2>Shortcodes</h2>
                <p class="sub">Use <code>[partnerads]</code> in any post or page.</p>
                <table class="mp-log" style="margin-top:0">
                    <thead><tr><th>Layout</th><th>Example</th></tr></thead>
                    <tbody>
                        <tr><td><strong>Grid</strong></td><td><code class="sc">[partnerads tag="Proteinpulver" layout="grid" columns="3" limit="6"]</code></td></tr>
                        <tr><td><strong>Table</strong></td><td><code class="sc">[partnerads tag="Sko" layout="table" limit="10"]</code></td></tr>
                        <tr><td><strong>Box</strong></td><td><code class="sc">[partnerads sku="12345" layout="box"]</code></td></tr>
                        <tr><td><strong>List</strong></td><td><code class="sc">[partnerads tag="Baby" layout="list" limit="5"]</code></td></tr>
                    </tbody>
                </table>
                <p class="sub" style="margin-top:14px;margin-bottom:4px"><strong>Filter options:</strong> <code>tag</code> <code>brand</code> <code>search</code> <code>sku</code> <code>limit</code> <code>columns</code> <code>orderby</code> <code>order</code> <code>button</code></p>

                <?php if(!empty($tags) && !is_wp_error($tags)): ?>
                <p class="sub" style="margin-top:12px;margin-bottom:4px"><strong>Your tags:</strong></p>
                <div class="mp-tags">
                    <?php foreach($tags as $tag): ?>
                        <span class="mp-tag"><?php echo esc_html($tag->name); ?> <span style="color:#787c82">(<?php echo $tag->count; ?>)</span></span>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>

            <?php if(!empty($log)): ?>
            <div class="mp-sec">
                <h2>Import History</h2>
                <table class="mp-log">
                    <thead><tr><th>Date</th><th>Feed</th><th>Total</th><th>New</th><th>Updated</th><th>Errors</th></tr></thead>
                    <tbody>
                    <?php foreach(array_reverse(array_slice($log,-10)) as $e): ?>
                    <tr>
                        <td><?php echo esc_html($e['time']); ?></td>
                        <td style="max-width:200px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;font-size:11px;font-family:monospace"><?php echo esc_html($e['feed']); ?></td>
                        <td><?php echo (int)$e['stats']['total']; ?></td>
                        <td><span class="mp-badge mp-bg"><?php echo (int)$e['stats']['created']; ?></span></td>
                        <td><span class="mp-badge mp-bb"><?php echo (int)$e['stats']['updated']; ?></span></td>
                        <td><?php echo $e['stats']['errors']>0?'<span class="mp-badge mp-br">'.(int)$e['stats']['errors'].'</span>':'0'; ?></td>
                    </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>

        <?php elseif($tab==='design'): ?>

            <!-- GLOBAL -->
            <div class="mp-sec">
                <h2>Global</h2>
                <p class="sub">Shared colors and button settings used by all layouts.</p>
                <div class="mp-g3">
                    <div class="mp-r"><label>Button Text</label><input type="text" id="s-button_text" value="<?php echo esc_attr($d['button_text']); ?>" style="font-family:inherit"></div>
                    <div class="mp-r"><label>Button Radius (px)</label><input type="number" id="s-btn_radius" value="<?php echo intval($d['btn_radius']); ?>" min="0" max="50" style="font-family:inherit"><p class="h">0 = square, 20+ = pill</p></div>
                    <div></div>
                </div>
                <div class="mp-g4" style="margin-bottom:14px">
                    <div class="mp-color"><input type="color" id="s-btn_color" value="<?php echo esc_attr($d['btn_color']); ?>"><span>Button</span></div>
                    <div class="mp-color"><input type="color" id="s-btn_text_color" value="<?php echo esc_attr($d['btn_text_color']); ?>"><span>Button Text</span></div>
                    <div class="mp-color"><input type="color" id="s-price_color" value="<?php echo esc_attr($d['price_color']); ?>"><span>Price</span></div>
                    <div class="mp-color"><input type="color" id="s-card_border" value="<?php echo esc_attr($d['card_border']); ?>"><span>Border</span></div>
                </div>
                <div class="mp-g4" style="margin-bottom:4px">
                    <div class="mp-ch" style="margin:0"><input type="checkbox" id="s-show_brand" <?php checked($d['show_brand']); ?>><label>Show Brand</label></div>
                    <div class="mp-ch" style="margin:0"><input type="checkbox" id="s-show_image" <?php checked($d['show_image']); ?>><label>Show Image</label></div>
                    <div class="mp-ch" style="margin:0"><input type="checkbox" id="s-show_arrow" <?php checked($d['show_arrow']); ?>><label>Arrow in Button</label></div>
                    <div></div>
                </div>
            </div>

            <!-- GRID -->
            <div class="mp-sec">
                <h2>Grid Layout</h2>
                <p class="sub"><code class="sc">[partnerads layout="grid"]</code></p>
                <div class="mp-g4">
                    <div class="mp-r"><label>Name Size (px)</label><input type="number" id="s-grid_name_size" value="<?php echo intval($d['grid_name_size']); ?>" min="10" max="24"></div>
                    <div class="mp-r"><label>Price Size (px)</label><input type="number" id="s-grid_price_size" value="<?php echo intval($d['grid_price_size']); ?>" min="10" max="30"></div>
                    <div class="mp-r"><label>Button Size (px)</label><input type="number" id="s-grid_btn_size" value="<?php echo intval($d['grid_btn_size']); ?>" min="10" max="20"></div>
                    <div class="mp-ch" style="margin:0;padding-top:24px"><input type="checkbox" id="s-grid_shadow" <?php checked($d['grid_shadow']); ?>><label>Hover Shadow</label></div>
                </div>
                <div class="mp-prev">
                    <h3>Preview</h3>
                    <div id="pv-grid" style="display:grid;grid-template-columns:repeat(3,1fr);gap:14px"></div>
                </div>
            </div>

            <!-- TABLE -->
            <div class="mp-sec">
                <h2>Table Layout</h2>
                <p class="sub"><code class="sc">[partnerads layout="table"]</code></p>
                <div class="mp-g4">
                    <div class="mp-r"><label>Name Size (px)</label><input type="number" id="s-table_name_size" value="<?php echo intval($d['table_name_size']); ?>" min="10" max="24"></div>
                    <div class="mp-r"><label>Price Size (px)</label><input type="number" id="s-table_price_size" value="<?php echo intval($d['table_price_size']); ?>" min="10" max="30"></div>
                    <div class="mp-r"><label>Button Size (px)</label><input type="number" id="s-table_btn_size" value="<?php echo intval($d['table_btn_size']); ?>" min="10" max="20"></div>
                    <div class="mp-r"><label>Row Padding (px)</label><input type="number" id="s-table_row_padding" value="<?php echo intval($d['table_row_padding']); ?>" min="4" max="24"></div>
                </div>
                <div class="mp-g4">
                    <div class="mp-r"><label>Button Style</label>
                        <select id="s-table_btn_style">
                            <option value="pill" <?php selected($d['table_btn_style'],'pill'); ?>>Pill</option>
                            <option value="square" <?php selected($d['table_btn_style'],'square'); ?>>Square</option>
                            <option value="link" <?php selected($d['table_btn_style'],'link'); ?>>Text link</option>
                        </select>
                    </div>
                    <div class="mp-r"><label>Border Style</label>
                        <select id="s-table_border">
                            <option value="horizontal" <?php selected($d['table_border'],'horizontal'); ?>>Horizontal lines</option>
                            <option value="full" <?php selected($d['table_border'],'full'); ?>>Full grid</option>
                            <option value="outer" <?php selected($d['table_border'],'outer'); ?>>Outer only</option>
                            <option value="none" <?php selected($d['table_border'],'none'); ?>>No borders</option>
                        </select>
                    </div>
                    <div class="mp-ch" style="margin:0;padding-top:24px"><input type="checkbox" id="s-table_show_header" <?php checked($d['table_show_header']); ?>><label>Header Row</label></div>
                    <div class="mp-ch" style="margin:0;padding-top:24px"><input type="checkbox" id="s-table_striped" <?php checked($d['table_striped']); ?>><label>Striped Rows</label></div>
                </div>
                <div class="mp-g3">
                    <div class="mp-ch" style="margin:0"><input type="checkbox" id="s-table_hover" <?php checked($d['table_hover']); ?>><label>Hover Effect</label></div>
                    <div class="mp-color"><input type="color" id="s-table_hover_color" value="<?php echo esc_attr($d['table_hover_color']); ?>"><span>Hover Color</span></div>
                    <div></div>
                </div>
                <div class="mp-prev">
                    <h3>Preview</h3>
                    <div id="pv-table"></div>
                </div>
            </div>

            <!-- BOX -->
            <div class="mp-sec">
                <h2>Box Layout</h2>
                <p class="sub"><code class="sc">[partnerads layout="box" sku="..."]</code></p>
                <div class="mp-g4">
                    <div class="mp-r"><label>Name Size (px)</label><input type="number" id="s-box_name_size" value="<?php echo intval($d['box_name_size']); ?>" min="14" max="30"></div>
                    <div class="mp-r"><label>Price Size (px)</label><input type="number" id="s-box_price_size" value="<?php echo intval($d['box_price_size']); ?>" min="14" max="36"></div>
                    <div class="mp-r"><label>Image Size (px)</label><input type="number" id="s-box_img_size" value="<?php echo intval($d['box_img_size']); ?>" min="80" max="300"></div>
                    <div class="mp-r"><label>Desc Lines</label><input type="number" id="s-box_desc_lines" value="<?php echo intval($d['box_desc_lines']); ?>" min="0" max="10"><p class="h">0 = hide</p></div>
                </div>
                <div class="mp-prev">
                    <h3>Preview</h3>
                    <div id="pv-box"></div>
                </div>
            </div>

            <!-- LIST -->
            <div class="mp-sec">
                <h2>List Layout</h2>
                <p class="sub"><code class="sc">[partnerads layout="list"]</code></p>
                <div class="mp-g3">
                    <div class="mp-r"><label>Name Size (px)</label><input type="number" id="s-list_name_size" value="<?php echo intval($d['list_name_size']); ?>" min="10" max="24"></div>
                    <div class="mp-r"><label>Price Size (px)</label><input type="number" id="s-list_price_size" value="<?php echo intval($d['list_price_size']); ?>" min="10" max="30"></div>
                    <div class="mp-ch" style="margin:0;padding-top:24px"><input type="checkbox" id="s-list_show_btn" <?php checked($d['list_show_btn']); ?>><label>Show Button</label></div>
                </div>
                <div class="mp-prev">
                    <h3>Preview</h3>
                    <div id="pv-list"></div>
                </div>
            </div>

            <div class="mp-acts">
                <button class="mp-btn mp-bp" id="mp-save-design" style="padding:10px 28px;font-size:14px">Save All Design Settings</button>
            </div>

        <?php endif; ?>
        </div>
    </div>

    <script>
    jQuery(function($){
        var nonce=<?php echo wp_json_encode(wp_create_nonce('mbpa_nonce')); ?>;
        var $st=$('#mp-status');

        var samples=[
            {name:'Whey 100 ISO Pure (750 g) – Chocolate',brand:'BODYLAB',price:'299',old:'349',currency:'DKK',desc:'Premium whey isolate med høj proteinkoncentration og lav fedtindhold.'},
            {name:'Protein Bar Crunchy (12x 64 g)',brand:'BODYLAB',price:'249',old:'',currency:'DKK',desc:'Sprød proteinbar med chokoladeovertrk og karamelkerne.'},
            {name:'ZMA Kapsler (120 stk)',brand:'BODYLAB',price:'149',old:'179',currency:'DKK',desc:'Zink, magnesium og B6 vitamin til normal muskelfunktion.'},
            {name:'Creatine Monohydrate (300 g)',brand:'BODYLAB',price:'129',old:'',currency:'DKK',desc:'Ren kreatin monohydrat til øget styrke og power.'}
        ];

        function v(id){var el=$('#s-'+id);if(!el.length)return '';return el.is(':checkbox')?el.is(':checked'):el.val();}

        function renderAll(){
            var bc=v('btn_color')||'#2d2d2d', btc=v('btn_text_color')||'#fff', br=v('btn_radius')||6, pc=v('price_color')||'#d63031', cb=v('card_border')||'#e5e5e5';
            var txt=v('button_text')||'Se produkt', arrow=v('show_arrow'), si=v('show_image'), sb=v('show_brand');
            var arw=arrow?' <svg style="width:11px;height:11px;fill:none;stroke:currentColor;stroke-width:2.5;vertical-align:middle;margin-left:3px" viewBox="0 0 24 24"><path d="M5 12h14M12 5l7 7-7 7"/></svg>':'';

            // GRID
            var gns=v('grid_name_size')||14, gps=v('grid_price_size')||16, gbs=v('grid_btn_size')||13, gsh=v('grid_shadow');
            var gh='';
            for(var i=0;i<3;i++){var p=samples[i];
                gh+='<div style="border:1px solid '+cb+';border-radius:8px;overflow:hidden;background:#fff;display:flex;flex-direction:column;transition:box-shadow .2s'+(gsh?';box-shadow:0 1px 4px rgba(0,0,0,.04)':'')+'">';
                if(si)gh+='<div style="background:#f8f8f8;height:100px;display:flex;align-items:center;justify-content:center;font-size:11px;color:#bbb;padding:8px">Image</div>';
                gh+='<div style="padding:12px;flex:1;display:flex;flex-direction:column">';
                if(sb)gh+='<div style="font-size:10px;font-weight:600;color:#888;text-transform:uppercase;margin-bottom:3px">'+p.brand+'</div>';
                gh+='<div style="font-size:'+gns+'px;font-weight:600;color:#1a1a1a;margin-bottom:6px;line-height:1.3">'+p.name+'</div>';
                gh+='<div style="font-size:'+gps+'px;font-weight:700;color:'+pc+';margin-bottom:auto;padding-bottom:10px">'+p.price+' '+p.currency;
                if(p.old)gh+=' <span style="font-size:'+Math.max(10,gps-3)+'px;color:#999;text-decoration:line-through;font-weight:400;margin-left:4px">'+p.old+' '+p.currency+'</span>';
                gh+='</div>';
                gh+='<a href="#" onclick="return false" style="display:block;text-align:center;padding:9px 14px;background:'+bc+';color:'+btc+';border-radius:'+br+'px;font-size:'+gbs+'px;font-weight:600;text-decoration:none">'+txt+arw+'</a>';
                gh+='</div></div>';
            }
            $('#pv-grid').html(gh);

            // TABLE
            var tns=v('table_name_size')||14, tps=v('table_price_size')||16, tbs=v('table_btn_size')||12, trp=v('table_row_padding')||12;
            var tbst=v('table_btn_style')||'pill', tbrd=v('table_border')||'horizontal';
            var thdr=v('table_show_header'), tstr=v('table_striped'), thov=v('table_hover'), thc=v('table_hover_color')||'#f8f8f8';
            var tbtnBg=(tbst==='link')?'transparent':bc, tbtnClr=(tbst==='link')?bc:btc, tbtnTd=(tbst==='link')?'underline':'none';
            var tbtnBr=(tbst==='pill')?'20px':(tbst==='link'?'0':'4px');

            var cellBorder='none', outerBorder='none';
            if(tbrd==='horizontal') cellBorder='border-bottom:1px solid '+cb;
            else if(tbrd==='full') cellBorder='border:1px solid '+cb;
            else if(tbrd==='outer') outerBorder='border:1px solid '+cb;

            var th='<table style="width:100%;border-collapse:collapse;font-size:14px;'+outerBorder+'">';
            if(thdr)th+='<thead><tr>'+(si?'<th style="padding:'+trp+'px;text-align:left;font-size:11px;font-weight:600;color:#888;text-transform:uppercase;'+cellBorder+'"></th>':'')+'<th style="padding:'+trp+'px;text-align:left;font-size:11px;font-weight:600;color:#888;text-transform:uppercase;'+cellBorder+'">Produkt</th><th style="padding:'+trp+'px;text-align:left;font-size:11px;font-weight:600;color:#888;text-transform:uppercase;'+cellBorder+'">Brand</th><th style="padding:'+trp+'px;text-align:left;font-size:11px;font-weight:600;color:#888;text-transform:uppercase;'+cellBorder+'">Pris</th><th style="padding:'+trp+'px;'+cellBorder+'"></th></tr></thead>';
            th+='<tbody>';
            for(var i=0;i<4;i++){var p=samples[i];
                var bg=(tstr&&i%2===1)?'#f9f9f9':'transparent';
                var hovAttr=thov?' onmouseenter="this.style.background=\''+thc+'\'" onmouseleave="this.style.background=\''+bg+'\'"':'';
                th+='<tr style="background:'+bg+'"'+hovAttr+'>';
                if(si)th+='<td style="padding:'+trp+'px;vertical-align:middle;width:50px;'+cellBorder+'"><div style="width:40px;height:40px;background:#f5f5f5;border-radius:4px;display:flex;align-items:center;justify-content:center;font-size:9px;color:#bbb">IMG</div></td>';
                th+='<td style="padding:'+trp+'px;vertical-align:middle;'+cellBorder+'"><div style="font-size:'+tns+'px;font-weight:600;color:#1a1a1a">'+p.name+'</div></td>';
                th+='<td style="padding:'+trp+'px;vertical-align:middle;'+cellBorder+'"><div style="font-size:11px;font-weight:600;color:#888;text-transform:uppercase">'+p.brand+'</div></td>';
                th+='<td style="padding:'+trp+'px;vertical-align:middle;'+cellBorder+'"><div style="font-size:'+tps+'px;font-weight:700;color:'+pc+';white-space:nowrap">'+p.price+' '+p.currency+'</div>';
                if(p.old)th+='<div style="font-size:'+Math.max(10,tps-4)+'px;color:#999;text-decoration:line-through">'+p.old+' '+p.currency+'</div>';
                th+='</td>';
                th+='<td style="padding:'+trp+'px;vertical-align:middle;'+cellBorder+'"><a href="#" onclick="return false" style="display:inline-flex;align-items:center;gap:4px;background:'+tbtnBg+';color:'+tbtnClr+';text-decoration:'+tbtnTd+';padding:'+(tbst==='link'?'4px 0':'8px 16px')+';border-radius:'+tbtnBr+';font-size:'+tbs+'px;font-weight:600;white-space:nowrap">'+txt+arw+'</a></td>';
                th+='</tr>';
            }
            th+='</tbody></table>';
            $('#pv-table').html(th);

            // BOX
            var bns=v('box_name_size')||18, bps=v('box_price_size')||20, bis=v('box_img_size')||160, bdl=v('box_desc_lines')||3;
            var bx='<div style="display:flex;gap:20px;background:#fff;border:1px solid '+cb+';border-radius:10px;padding:20px;align-items:center">';
            if(si)bx+='<div style="width:'+bis+'px;height:'+bis+'px;background:#f8f8f8;border-radius:8px;flex-shrink:0;display:flex;align-items:center;justify-content:center;font-size:12px;color:#bbb">Image</div>';
            bx+='<div style="flex:1;min-width:0">';
            if(sb)bx+='<div style="font-size:11px;font-weight:600;color:#888;text-transform:uppercase;margin-bottom:4px">'+samples[0].brand+'</div>';
            bx+='<div style="font-size:'+bns+'px;font-weight:700;color:#1a1a1a;margin-bottom:6px">'+samples[0].name+'</div>';
            if(bdl>0)bx+='<p style="font-size:13px;color:#555;line-height:1.5;margin:0 0 12px;display:-webkit-box;-webkit-line-clamp:'+bdl+';-webkit-box-orient:vertical;overflow:hidden">'+samples[0].desc+'</p>';
            bx+='<div style="font-size:'+bps+'px;font-weight:700;color:'+pc+';margin-bottom:12px">'+samples[0].price+' '+samples[0].currency;
            if(samples[0].old)bx+=' <span style="font-size:'+Math.max(12,bps-4)+'px;color:#999;text-decoration:line-through;margin-left:6px">'+samples[0].old+' '+samples[0].currency+'</span>';
            bx+='</div>';
            bx+='<a href="#" onclick="return false" style="display:inline-flex;align-items:center;gap:4px;padding:10px 20px;background:'+bc+';color:'+btc+';border-radius:'+br+'px;font-size:13px;font-weight:600;text-decoration:none">'+txt+arw+'</a>';
            bx+='</div></div>';
            $('#pv-box').html(bx);

            // LIST
            var lns=v('list_name_size')||14, lps=v('list_price_size')||15, lsb=v('list_show_btn');
            var lh='<ul style="margin:0;padding:0;list-style:none">';
            for(var i=0;i<4;i++){var p=samples[i];
                lh+='<li style="display:flex;align-items:center;justify-content:space-between;padding:10px 0;border-bottom:1px solid #f0f0f0;gap:12px">';
                lh+='<div style="flex:1;min-width:0"><div style="font-size:'+lns+'px;font-weight:500;color:#1a1a1a">'+p.name+'</div>';
                if(sb)lh+='<div style="font-size:11px;color:#888;font-weight:600;text-transform:uppercase">'+p.brand+'</div>';
                lh+='</div>';
                lh+='<span style="font-size:'+lps+'px;font-weight:700;color:'+pc+';white-space:nowrap">'+p.price+' '+p.currency+'</span>';
                if(lsb)lh+='<a href="#" onclick="return false" style="display:inline-flex;align-items:center;background:'+bc+';color:'+btc+';padding:6px 14px;border-radius:'+br+'px;font-size:12px;font-weight:600;white-space:nowrap;text-decoration:none">'+txt+'</a>';
                lh+='</li>';
            }
            lh+='</ul>';
            $('#pv-list').html(lh);
        }

        $('[id^="s-"]').on('input change',renderAll);
        if($('#pv-grid').length) renderAll();

        function collectFields(){
            var data={action:'mbpa_save',nonce:nonce};
            $('[id^="s-"]').each(function(){
                var key=$(this).attr('id').replace('s-','');
                if($(this).is(':checkbox')) data[key]=$(this).is(':checked')?1:0;
                else data[key]=$(this).val();
            });
            return data;
        }

        function saveBtn($b, extra){
            var orig=$b.text();
            $b.prop('disabled',true).text('Saving...');
            var data=collectFields();
            if(extra) $.extend(data,extra);
            $.post(ajaxurl,data,function(r){
                $b.prop('disabled',false);
                if(r.success){$b.text('Saved ✓');setTimeout(function(){$b.text(orig)},2000)}
                else{alert(r.data);$b.text(orig)}
            }).fail(function(){$b.prop('disabled',false).text(orig);alert('Failed')});
        }

        $('#mp-save-feeds').on('click',function(){saveBtn($(this),{feeds:$('#mp-feeds').val(),auto_sync:$('#mp-sync').is(':checked')?1:0})});
        $('#mp-save-design').on('click',function(){saveBtn($(this))});

        function firstUrl(){var ls=$('#mp-feeds').val().trim().split('\n');for(var i=0;i<ls.length;i++){var u=ls[i].trim();if(u&&u.indexOf('http')===0)return u}return ''}

        $('#mp-test').on('click',function(){
            var u=firstUrl();if(!u){alert('Add a feed URL');return}
            var $b=$(this).prop('disabled',true).text('Loading...');
            $st.html('<span class="mp-spin"></span> Fetching...');$('#mp-preview').html('');
            $.post(ajaxurl,{action:'mbpa_preview',nonce:nonce,url:u},function(r){
                $b.prop('disabled',false).text('Preview Feed');
                if(r.success){$st.html('<strong>'+r.data.total+'</strong> products found');var h='<table class="mp-ptable"><thead><tr><th></th><th>Name</th><th>Price</th><th>Brand</th><th>Category</th></tr></thead><tbody>';r.data.sample.forEach(function(p){h+='<tr><td>'+(p.image?'<img src="'+p.image+'">':'')+'</td><td>'+(p.name||'-')+'</td><td>'+(p.price||'-')+' '+(p.currency||'')+'</td><td>'+(p.brand||'-')+'</td><td>'+(p.category||'-')+'</td></tr>'});h+='</tbody></table>';$('#mp-preview').html(h)}
                else $st.html('<span style="color:#d63638">'+r.data+'</span>')
            }).fail(function(){$b.prop('disabled',false).text('Preview Feed');$st.html('<span style="color:#d63638">Failed</span>')});
        });

        $('#mp-import').on('click',function(){
            if(!$('#mp-feeds').val().trim()){alert('Add feed URLs');return}
            if(!confirm('Import all products?'))return;
            var $b=$(this).prop('disabled',true).html('<span class="mp-spin"></span> Importing...');
            $st.html('<span class="mp-spin"></span> Importing — may take a few minutes...');
            $.post(ajaxurl,{action:'mbpa_import',nonce:nonce},function(r){
                $b.prop('disabled',false).text('Import All');
                if(r.success){var s=r.data;$st.html('<div style="background:#d4edda;border:1px solid #00a32a;padding:12px;border-radius:4px">Done — <b>'+s.total+'</b> products from <b>'+s.feeds+'</b> feed(s). New: <b>'+s.created+'</b>, Updated: <b>'+s.updated+'</b></div>');setTimeout(function(){location.reload()},2500)}
                else $st.html('<span style="color:#d63638">'+r.data+'</span>')
            }).fail(function(){$b.prop('disabled',false).text('Import All');$st.html('<span style="color:#d63638">Failed or timed out</span>')});
        });

        $('#mp-del').on('click',function(){
            if(!confirm('Delete ALL imported products?'))return;
            $(this).prop('disabled',true).text('Deleting...');
            $.post(ajaxurl,{action:'mbpa_delete_all',nonce:nonce},function(r){
                if(r.success){alert(r.data.deleted+' deleted');location.reload()}else alert(r.data)
            });
        });
    });
    </script>
    <?php
}

// =====================================================
// SHORTCODE + FRONTEND CSS
// =====================================================

add_shortcode('partnerads','mbpa_shortcode');
function mbpa_shortcode($atts) {
    $atts = shortcode_atts(array(
        'layout'=>'grid','limit'=>6,'columns'=>3,
        'tag'=>'','brand'=>'','search'=>'','sku'=>'',
        'orderby'=>'date','order'=>'desc','button'=>'',
    ),$atts,'partnerads');

    $d   = mbpa_design();
    $btn = !empty($atts['button']) ? $atts['button'] : $d['button_text'];
    $arr = $d['show_arrow'] ? '<svg style="width:12px;height:12px;fill:none;stroke:currentColor;stroke-width:2.5;vertical-align:middle;margin-left:4px" viewBox="0 0 24 24"><path d="M5 12h14M12 5l7 7-7 7"/></svg>' : '';

    $args = array('post_type'=>'mbpa_product','posts_per_page'=>intval($atts['limit']),'post_status'=>'publish','order'=>strtoupper($atts['order'])==='ASC'?'ASC':'DESC');
    if($atts['orderby']==='price'){$args['meta_key']='_mbpa_price';$args['orderby']='meta_value_num';}
    elseif($atts['orderby']==='name'){$args['orderby']='title';}

    if(!empty($atts['sku'])){$args['meta_query'][]=array('key'=>'_mbpa_sku','value'=>array_map('trim',explode(',',$atts['sku'])),'compare'=>'IN');}
    if(!empty($atts['tag'])){$args['tax_query']=array(array('taxonomy'=>'mbpa_tag','field'=>'name','terms'=>array_map('trim',explode(',',$atts['tag']))));}
    if(!empty($atts['brand'])){$args['meta_query'][]=array('key'=>'_mbpa_brand','value'=>array_map('trim',explode(',',$atts['brand'])),'compare'=>'IN');}
    if(!empty($atts['search'])) $args['s']=$atts['search'];

    $q = new WP_Query($args);
    if(!$q->have_posts()) return '';

    $items = array();
    while($q->have_posts()){$q->the_post();$id=get_the_ID();
        $img=get_post_meta($id,'_mbpa_image',true);
        $items[]=array('name'=>get_the_title(),'desc'=>wp_trim_words(get_the_content(),25),'url'=>get_post_meta($id,'_mbpa_url',true),'image'=>$img,'price'=>get_post_meta($id,'_mbpa_price',true),'old'=>get_post_meta($id,'_mbpa_old_price',true),'currency'=>get_post_meta($id,'_mbpa_currency',true)?:'DKK','brand'=>get_post_meta($id,'_mbpa_brand',true));
    }
    wp_reset_postdata();
    if(empty($items)) return '';

    $bc  = esc_attr($d['btn_color']);
    $btc = esc_attr($d['btn_text_color']);
    $br  = intval($d['btn_radius']);
    $pc  = esc_attr($d['price_color']);
    $cb  = esc_attr($d['card_border']);
    $si  = $d['show_image'];
    $sb  = $d['show_brand'];
    $fmt = function($v){return number_format((float)$v,0,',','.');};

    // Per-layout sizes
    $gns=intval($d['grid_name_size']);$gps=intval($d['grid_price_size']);$gbs=intval($d['grid_btn_size']);
    $tns=intval($d['table_name_size']);$tps=intval($d['table_price_size']);$tbs=intval($d['table_btn_size']);
    $tbr=$d['table_btn_style']==='pill'?'20px':($d['table_btn_style']==='link'?'0':'4px');
    $tstr=$d['table_striped'];$thdr=$d['table_show_header'];
    $tbdr=$d['table_border'];$thov=$d['table_hover'];$thc=esc_attr($d['table_hover_color']);$trp=intval($d['table_row_padding']);
    $bns=intval($d['box_name_size']);$bps=intval($d['box_price_size']);$bis=intval($d['box_img_size']);$bdl=intval($d['box_desc_lines']);
    $lns=intval($d['list_name_size']);$lps=intval($d['list_price_size']);$lsb=$d['list_show_btn'];

    static $css_done=false;
    $o='';
    if(!$css_done){
        $o.='<style>
.pa-grid{display:grid;gap:16px;margin:24px 0}
.pa-grid[data-c="2"]{grid-template-columns:repeat(2,1fr)}.pa-grid[data-c="3"]{grid-template-columns:repeat(3,1fr)}.pa-grid[data-c="4"]{grid-template-columns:repeat(4,1fr)}
@media(max-width:768px){.pa-grid[data-c="3"],.pa-grid[data-c="4"]{grid-template-columns:repeat(2,1fr)}}@media(max-width:480px){.pa-grid{grid-template-columns:1fr!important}}
.pa-ci{background:#fff;border:1px solid '.$cb.';border-radius:8px;overflow:hidden;display:flex;flex-direction:column;transition:box-shadow .2s}'.($d['grid_shadow']?'.pa-ci:hover{box-shadow:0 4px 12px rgba(0,0,0,.07)}':'').'
.pa-img{width:100%;aspect-ratio:1;object-fit:contain;background:#f8f8f8;padding:12px}
.pa-bd{padding:14px;flex:1;display:flex;flex-direction:column}
.pa-br{font-size:11px;font-weight:600;color:#888;text-transform:uppercase;letter-spacing:.5px;margin-bottom:3px}
.pa-nm{font-size:'.$gns.'px;font-weight:600;color:#1a1a1a;margin:0 0 6px;line-height:1.3}
.pa-pr{font-size:'.$gps.'px;font-weight:700;color:'.$pc.';margin-bottom:auto;padding-bottom:12px}
.pa-old{font-size:'.max(10,$gps-3).'px;color:#999;text-decoration:line-through;font-weight:400;margin-left:6px}
.pa-btn{display:flex;align-items:center;justify-content:center;gap:4px;background:'.$bc.';color:'.$btc.'!important;text-decoration:none!important;padding:10px 16px;border-radius:'.$br.'px;font-size:'.$gbs.'px;font-weight:600;transition:background .15s;margin-top:auto}
.pa-btn:hover{opacity:.9;color:'.$btc.'!important}
.pa-tbl{width:100%;border-collapse:collapse;margin:24px 0;font-size:14px'.($tbdr==='outer'?';border:1px solid '.$cb:'').'}
.pa-tbl th{text-align:left;padding:'.$trp.'px;font-size:11px;font-weight:600;color:#888;text-transform:uppercase'.($tbdr==='horizontal'?';border-bottom:2px solid '.$cb:($tbdr==='full'?';border:1px solid '.$cb:'')).'}
.pa-tbl td{padding:'.$trp.'px;vertical-align:middle'.($tbdr==='horizontal'?';border-bottom:1px solid '.$cb:($tbdr==='full'?';border:1px solid '.$cb:'')).'}
'.($thov?'.pa-tbl tbody tr:hover td{background:'.$thc.'}':'').'
'.($tstr?'.pa-tbl tbody tr:nth-child(even) td{background:#f9f9f9}':'').'
.pa-ti{width:50px;height:50px;object-fit:contain;border-radius:6px;background:#f8f8f8}
.pa-tn{font-weight:600;color:#1a1a1a;font-size:'.$tns.'px}.pa-tb{font-size:11px;font-weight:600;color:#888;text-transform:uppercase}
.pa-tp{font-size:'.$tps.'px;font-weight:700;color:'.$pc.';white-space:nowrap}.pa-to{font-size:'.max(10,$tps-4).'px;color:#999;text-decoration:line-through}
.pa-tbtn{display:inline-flex;align-items:center;gap:4px;background:'.($d['table_btn_style']==='link'?'transparent':$bc).';color:'.($d['table_btn_style']==='link'?$bc:$btc).'!important;text-decoration:'.($d['table_btn_style']==='link'?'underline':'none').'!important;padding:'.($d['table_btn_style']==='link'?'4px 0':'8px 16px').';border-radius:'.$tbr.';font-size:'.$tbs.'px;font-weight:600;white-space:nowrap;transition:opacity .15s}
.pa-tbtn:hover{opacity:.85}
@media(max-width:640px){.pa-tbl th:nth-child(3),.pa-tbl td:nth-child(3){display:none}}
.pa-box{display:flex;gap:20px;background:#fff;border:1px solid '.$cb.';border-radius:10px;padding:20px;margin:24px 0;align-items:center}.pa-box:hover{box-shadow:0 4px 12px rgba(0,0,0,.05)}
.pa-bi{width:'.$bis.'px;height:'.$bis.'px;object-fit:contain;border-radius:8px;background:#f8f8f8;flex-shrink:0;padding:8px}
.pa-bb{flex:1;min-width:0}
.pa-bn{font-size:'.$bns.'px;font-weight:700;color:#1a1a1a;margin:0 0 6px}
.pa-bds{font-size:13px;color:#555;line-height:1.5;margin:0 0 12px;display:-webkit-box;-webkit-line-clamp:'.$bdl.';-webkit-box-orient:vertical;overflow:hidden}
.pa-bp{font-size:'.$bps.'px;font-weight:700;color:'.$pc.';margin-bottom:12px}
@media(max-width:640px){.pa-box{flex-direction:column;text-align:center}.pa-bi{width:100%;height:200px}}
.pa-list{margin:24px 0;list-style:none!important;padding:0!important}
.pa-list li{display:flex;align-items:center;justify-content:space-between;padding:10px 0;border-bottom:1px solid #f0f0f0;gap:12px}
.pa-list li:last-child{border-bottom:none}
.pa-li{flex:1;min-width:0}
.pa-ln{font-size:'.$lns.'px;font-weight:500}.pa-ln a{color:#1a1a1a!important;text-decoration:none!important}.pa-ln a:hover{text-decoration:underline!important}
.pa-lb{font-size:11px;color:#888;font-weight:600;text-transform:uppercase}
.pa-lp{font-size:'.$lps.'px;font-weight:700;color:'.$pc.';white-space:nowrap}
.pa-lbtn{display:inline-flex;align-items:center;background:'.$bc.';color:'.$btc.'!important;text-decoration:none!important;padding:6px 14px;border-radius:'.$br.'px;font-size:12px;font-weight:600;white-space:nowrap;transition:opacity .15s}
.pa-lbtn:hover{opacity:.9;color:'.$btc.'!important}
</style>';
        $css_done=true;
    }

    $rel='nofollow noopener sponsored';
    $ly=$atts['layout'];

    if($ly==='grid'){
        $c=max(1,min(4,intval($atts['columns'])));
        $o.='<div class="pa-grid" data-c="'.$c.'">';
        foreach($items as $p){
            $o.='<div class="pa-ci">';
            if($si&&$p['image']) $o.='<img class="pa-img" src="'.esc_url($p['image']).'" alt="'.esc_attr($p['name']).'" loading="lazy">';
            $o.='<div class="pa-bd">';
            if($sb&&$p['brand']) $o.='<div class="pa-br">'.esc_html($p['brand']).'</div>';
            $o.='<div class="pa-nm">'.esc_html($p['name']).'</div>';
            $o.='<div class="pa-pr">'.esc_html($fmt($p['price'])).' '.esc_html($p['currency']);
            if($p['old']) $o.=' <span class="pa-old">'.esc_html($fmt($p['old'])).' '.esc_html($p['currency']).'</span>';
            $o.='</div>';
            $o.='<a class="pa-btn" href="'.esc_url($p['url']).'" target="_blank" rel="'.$rel.'">'.esc_html($btn).' '.$arr.'</a>';
            $o.='</div></div>';
        }
        $o.='</div>';
    }
    elseif($ly==='table'){
        $o.='<table class="pa-tbl">';
        if($thdr) $o.='<thead><tr>'.($si?'<th></th>':'').'<th>Produkt</th><th>Brand</th><th>Pris</th><th></th></tr></thead>';
        $o.='<tbody>';
        foreach($items as $p){
            $o.='<tr>';
            if($si) $o.='<td>'.($p['image']?'<img class="pa-ti" src="'.esc_url($p['image']).'" loading="lazy">':'').'</td>';
            $o.='<td><div class="pa-tn">'.esc_html($p['name']).'</div></td>';
            $o.='<td><div class="pa-tb">'.esc_html($p['brand']).'</div></td>';
            $o.='<td><div class="pa-tp">'.esc_html($fmt($p['price'])).' '.esc_html($p['currency']).'</div>';
            if($p['old']) $o.='<div class="pa-to">'.esc_html($fmt($p['old'])).' '.esc_html($p['currency']).'</div>';
            $o.='</td>';
            $o.='<td><a class="pa-tbtn" href="'.esc_url($p['url']).'" target="_blank" rel="'.$rel.'">'.esc_html($btn).' '.$arr.'</a></td>';
            $o.='</tr>';
        }
        $o.='</tbody></table>';
    }
    elseif($ly==='box'){
        foreach($items as $p){
            $o.='<div class="pa-box">';
            if($si&&$p['image']) $o.='<img class="pa-bi" src="'.esc_url($p['image']).'" alt="'.esc_attr($p['name']).'" loading="lazy">';
            $o.='<div class="pa-bb">';
            if($sb&&$p['brand']) $o.='<div class="pa-br">'.esc_html($p['brand']).'</div>';
            $o.='<div class="pa-bn">'.esc_html($p['name']).'</div>';
            if($bdl>0&&$p['desc']) $o.='<p class="pa-bds">'.esc_html($p['desc']).'</p>';
            $o.='<div class="pa-bp">'.esc_html($fmt($p['price'])).' '.esc_html($p['currency']);
            if($p['old']) $o.=' <span class="pa-old">'.esc_html($fmt($p['old'])).' '.esc_html($p['currency']).'</span>';
            $o.='</div>';
            $o.='<a class="pa-btn" href="'.esc_url($p['url']).'" target="_blank" rel="'.$rel.'" style="max-width:200px">'.esc_html($btn).' '.$arr.'</a>';
            $o.='</div></div>';
        }
    }
    elseif($ly==='list'){
        $o.='<ul class="pa-list">';
        foreach($items as $p){
            $o.='<li><div class="pa-li"><div class="pa-ln"><a href="'.esc_url($p['url']).'" target="_blank" rel="'.$rel.'">'.esc_html($p['name']).'</a></div>';
            if($sb&&$p['brand']) $o.='<div class="pa-lb">'.esc_html($p['brand']).'</div>';
            $o.='</div><span class="pa-lp">'.esc_html($fmt($p['price'])).' '.esc_html($p['currency']).'</span>';
            if($lsb) $o.='<a class="pa-lbtn" href="'.esc_url($p['url']).'" target="_blank" rel="'.$rel.'">'.esc_html($btn).'</a>';
            $o.='</li>';
        }
        $o.='</ul>';
    }
    return $o;
}
